//
//  UserProfilesViewUseCase+ViewData.swift
//
//
//  Created by Nicolas Sabella on 20/01/2025.
//

import Foundation
import MapKit
import SwiftUI

import happnConfiguration
import happnConstants
import happnFlashnote
import happnPolis
import happnTraits
import happnUsers
import happnUtils
import XibLoc

typealias CardUserStateTokenAndId = (token: CardUserStateToken, id: String)

extension UserProfilesViewUseCase {
	//MARK: - User Infos
	internal func createUserInfoViewData(
		getTraitsUseCase: GetTraitsUseCaseProtocol,
		type: UserProfilesType,
		badgeType: UserBadgeType?,
		flashnote: FlashnoteDomainModel?,
		userMe: UserDomainModel,
		profile: UserDomainModel,
		isReactOnContentEnabled: Bool
	) -> UserProfileViewType {

		var bannerViewData: BannerViewData?
		var expandableBannerViewData: ExpandableBannerViewData?

		if romaBadgeUsecase.isEnabled, let traits = getTraitsUseCase.current() {
			expandableBannerViewData = createExpandableBannerViewData(
				getTraitsUseCase: getTraitsUseCase,
				traits: traits,
				userMe: userMe,
				profile: profile
			)
		} else {
			withAnimation {
				bannerViewData = createRomaBadgeViewData(
					type: type,
					badgeType: badgeType,
					flashnote: flashnote,
					userMe: userMe,
					profile: profile,
					isReactOnContentEnabled: isReactOnContentEnabled
				)
			}
		}

		return .userInfos(
			viewData: UserInfosViewData(
				otherUser: profile,
				userMe: userMe,
				superCrushMessage: flashnote?.preview,
				banner: bannerViewData,
				expandableBanner: expandableBannerViewData,
				badgeType: badgeType,
				romaV2Enabled: isMe ? false : romaBadgeUsecase.isEnabled,
				isFromConversation: type.isOpenProfileFromConversation
			)
		)
	}

	private func createExpandableBannerViewData(
		getTraitsUseCase: GetTraitsUseCaseProtocol,
		traits: [TraitDomainModel],
		userMe: UserDomainModel,
		profile: UserDomainModel
	) -> ExpandableBannerViewData? {
		guard !isMe,
			  let userMePictureURL = userMe.mainPicture?.url(),
			  let profilePictureURL = profile.mainPicture?.url()
		else { return nil }

		var pillsViewData: [PillViewData] = []

		if romaBadgeUsecase.withoutTraits {
			pillsViewData = hobbiesAndSpots(userMe: userMe, profile: profile)
		} else {
			let relationshipAnswersViewData = UserTraitMultiChoiceViewData
				.commonMultiChoiceTraitAnswers(
					trait: getTraitsUseCase.current(trait: .relationship),
					targetUser: profile,
					userMe: userMe
				)
				.map { PillViewData(text: $0, leftIcon: .localIcon(IconAsset.search)) }

			let traitsViewData = createCommonTraitsData(traits: traits, userMe: userMe, profile: profile)

			pillsViewData = hobbiesAndSpots(userMe: userMe, profile: profile) + relationshipAnswersViewData + traitsViewData

			if userMe.cityOfResidence?.city == profile.cityOfResidence?.city, let city = userMe.cityOfResidence?.city {
				pillsViewData.append(PillViewData(text: city, leftIcon: .localIcon(IconAsset.home)))
			}
		}

		let shouldDisplayExpandableBanner = !pillsViewData.isEmpty && pillsViewData.count >= romaBadgeUsecase.minimumAffinitiesNumber

		commonPointsCount = shouldDisplayExpandableBanner ? pillsViewData.count : 0

		return shouldDisplayExpandableBanner ? ExpandableBannerViewData(
			avatarSetViewData: AvatarSetViewData([
				AvatarIconViewData(
					type: .picture(ImageSource.remote(userMePictureURL)),
					showBorder: true
				),
				AvatarIconViewData(
					type: .picture(ImageSource.remote(profilePictureURL)),
					showBorder: true
				)]),
			title: L10n.romaSectionTitle,
			pillViewData: pillsViewData
		) : nil
	}

	//MARK: - Roma Badge
	internal func createRomaBadgeViewData(
		type: UserProfilesType,
		badgeType: UserBadgeType?,
		flashnote: FlashnoteDomainModel?,
		userMe: UserDomainModel,
		profile: UserDomainModel,
		isReactOnContentEnabled: Bool
	) -> BannerViewData? {
		guard !isMe else { return nil }

		var message: String?

		if badgeType == .cometExplanation, let cometExplanation = profile.cometExplanation, !cometExplanation.isEmpty {
			message = cometExplanation
		} else if badgeType == .comet {
			message = nil
		} else {
			message = flashnote?.preview
		}

		var avatarSetIconViewTypes: [AvatarIconViewType] = []
		if let userMeURL = userMe.mainPicture?.url(), let otherUserMeURL = profile.mainPicture?.url() {
			avatarSetIconViewTypes = [
				.picture(.remote(userMeURL)),
				.picture(.remote(otherUserMeURL))
			]
		}

			guard let badgeType, let bannerViewData = UserBadgeViewData(
				badgeType: badgeType,
				meIsMale: userMe.isLocalizedMale,
				otherIsMale: profile.isLocalizedMale,
				avatarSetIconViewTypes: avatarSetIconViewTypes,
				message: message,
				isReactOnContentEnabled: isReactOnContentEnabled
			).bannerViewData else { return nil }

		switch type {
			case .vibes:
				let isRelationshipBadge = badgeType == .likedMe || badgeType == .charmedMe
				return isRelationshipBadge ? bannerViewData : nil

			case .timeline, .openProfile, .flashnote:
				return bannerViewData
		}
	}

	// MARK: - Photos List
	internal func createPhotosViewData(
		type: UserProfilesType,
		badgeType: UserBadgeType?,
		flashnote: FlashnoteDomainModel?,
		userMe: UserDomainModel,
		profile: UserDomainModel
	) -> [CardUserStateTokenAndId] {
		var picturesViewData: [CardUserStateTokenAndId] = []
		let pictures = (isMe ? userMe : profile).pictures?.compactMap { $0 } ?? []
		let picturesURL = (isMe ? userMe : profile).pictures?.compactMap { $0.url() } ?? []

		if picturesURL.isEmpty {
			// When a profile doesn't have any pic for some reasons
			// add CardUser with empty token when rebase
		} else {
			for i in 0 ... pictures.count - 1 {
				let photoViewData = CardUserWithPictureViewData(
					image: .remote(picturesURL[i]),
					showBorder: false
				)
				let stateToken: CardUserStateToken = .image(viewData: photoViewData)
				picturesViewData.append((stateToken, pictures[i].id))
			}
		}

		return picturesViewData
	}

	// MARK: - Relationship + User infos + Traits
	internal func createRelationshipUserInfosAndTraitsViewData(
		type: UserProfilesType,
		relationshipTrait: TraitDomainModel?,
		traits: [TraitDomainModel],
		userMe: UserDomainModel,
		profile: UserDomainModel
	) -> UserProfileViewType? {
		let relationshipViewData = UserTraitMultiChoiceViewData.getMultiChoiceAnswerViewData(
			targetUser: profile,
			userMe: userMe,
			trait: relationshipTrait,
			showCommun: !isMe
		)

		let profileInfoViewDatas = createUserProfileInfoViewDatas(profile: profile, userMe: userMe)
		let userTraitsList = profileInfoViewDatas + createUserTraitListViewDatas(
			traits: traits,
			userMe: userMe,
			profile: profile,
			showCommun: !isMe
		)

		var sections: [CardMultiSelectionSectionViewData] = []
		if let cardMultiSelectionViewData = relationshipViewData?.cardMultiSelectionViewData {
			sections.append(cardMultiSelectionViewData)
		}

		if userTraitsList.isNotEmpty {
			sections.append(
				CardMultiSelectionSectionViewData(
					title: L10n.commonAboutMe,
					pills: userTraitsList
				)
			)
		}

		guard sections.isNotEmpty else { return nil }
		return .relationshipUserInfosAndTraits(viewData: CardMultiSelectionViewData(sections: sections))
	}

	// MARK: - Teasers
	internal func createUserInputAnswerViewData(
		type: UserProfilesType,
		userMe: UserDomainModel,
		profile: UserDomainModel,
		traits: [TraitDomainModel]?,
		index: Int,
		isReplyToTeasersEnabled: Bool
	) -> UserProfileViewType? {
		guard let userInputs = (profile.traits?.filter { $0.type == .userInput }) else { return nil }

		guard let userInput = userInputs.first,
			  case let .userInput(answers) = userInput.answer,
			  let answer = answers[safe: index],
			  let trait = (traits?.first { $0.id == userInput.traitId }),
			  case let .userInput(categories) = trait.options
		else { return nil }

		let options = categories.flatMap { $0.options }
		guard let option = (options.first { $0.id == answer.optionId }) else { return nil }

		let isCommun = userMe.getUserInputTraitAnswer(traitId: trait.id, optionId: answer.optionId) != nil && !isMe

		var reactionButtonViewData: ButtonViewData?
		let alreadyCrushed = profile.relationToMe == .mutual(date: nil)
		if isReplyToTeasersEnabled && !alreadyCrushed && !(profile.isCharmed ?? false) && !isMe {
			reactionButtonViewData = ButtonViewData(
				title: L10n.replyToTeaserNpdButton,
				rightLottieComponentViewData: LottieComponentViewData(
					type: .json(.reactionIcon),
					loopMode: .repeat(3),
					backgroundBehavior: .pauseAndRestore,
					backgroundColor: .sky,
					contentModeToken: .scaleAspectFit
				)
			)
		}

		let viewData = UserTraitUserInputAnswerViewData(
			traitId: trait.id,
			option: option,
			answer: answer,
			promptViewData: CardPromptViewData(
				avatar: .init(
					type: .icon(IconAsset.quoteOpen),
					overridedBackgroundColor: .fillToken(isCommun ? .lavender : nil)
				),
				title: option.text,
				description: answer.answerText,
				plainButtonViewData: reactionButtonViewData
			)
		)

		return .userInputAnswer(viewData: viewData)
	}

	// MARK: - Hobbies
	internal func createHobbiesViewData(
		type: UserProfilesType,
		traitUseCase: GetTraitsUseCaseProtocol,
		userMe: UserDomainModel,
		profile: UserDomainModel
	) -> UserProfileViewType? {
		guard let hobbiesTraitViewData = UserTraitMultiChoiceViewData.getMultiChoiceAnswerViewData(
			targetUser: profile,
			userMe: userMe,
			trait: traitUseCase.current(trait: .hobbies),
			showCommun: !isMe
		) else { return nil }

		return .traitMultiChoice(viewData: hobbiesTraitViewData)
	}

	// MARK: - Spots
	internal func createSpotsViewData(
		type: UserProfilesType,
		spotsIsEnabled: Bool,
		userMe: UserDomainModel,
		profile: UserDomainModel
	) -> UserProfileViewType? {
		guard spotsIsEnabled, let userSpots = profile.spots,
			  userSpots.isNotEmpty else { return nil }

		let pillViewDatas = userSpots.map { spot in
			PillViewData(
				text: spot.name,
				avatarViewData: getSpotCategoryAvatarIconViewData(for: spot.category),
				state: (userMe.spots?.contains(spot) ?? false) && !isMe ? .isActive : .default
			)
		}

		return .spots(
			viewData: CardPillsViewData(
				title: L10n.dailyProfileViewSpotsTitle,
				pillViewDatas: pillViewDatas,
				pillNumberOfLines: 1
			)
		)
	}

	// MARK: - Description
	internal func createDescriptionViewData(
		type: UserProfilesType,
		profile: UserDomainModel
	) -> UserProfileViewType? {
		guard let descriptionText = profile.description, !descriptionText.isEmpty else { return nil }

		let userDescriptionViewData = CardTextViewData(
			title: L10n.dailyProfileViewDescriptionSectionTitle,
			text: descriptionText,
			linksAllowed: profile.clickableMessageLink || profile.type == .sponsor,
			showBorder: true
		)

		return .description(viewData: userDescriptionViewData)
	}

	// MARK: - Map
	internal func createMapViewData(
		type: UserProfilesType,
		userMe: UserDomainModel,
		profile: UserDomainModel,
		profileType: TimelineProfileRecommendationTypeDomainModel?,
		placemark: CLPlacemark?
	) -> UserProfileViewType? {
		guard let mapViewData = setupMapViewData(
			userMe: userMe,
			profile: profile,
			profileType: profileType,
			placeMark: placemark
		) else { return nil }

		return .crossing(viewData: mapViewData)
	}

	// MARK: - Block & Report
	internal func createBlockReportViewData(
		type: UserProfilesType,
		profile: UserDomainModel
	) -> UserProfileViewType? {
		guard !profile.isModerator, !isMe else { return nil }

		let blockReportViewData = ButtonViewData(
			title: L10n.profileBlockAndReportCta,
			overridedBackgroundColor: .alpha06
		)
		return .blockReport(viewData: blockReportViewData)
	}

	// MARK: - Utils
	internal func addPictureIfAvailable(
		pictureIndex: Int,
		picturesViewData: [CardUserStateTokenAndId]
	) -> UserProfileViewType? {
		if let carousel = addCarouselIfAvailable(
			pictureIndex: pictureIndex,
			picturesViewData: picturesViewData
		) {
			return carousel
		} else {
			guard let viewData = picturesViewData[safe: pictureIndex] else { return nil }
			return .photo(viewData: viewData.token, id: viewData.id)
		}
	}

	private func addCarouselIfAvailable(
		pictureIndex: Int,
		picturesViewData: [CardUserStateTokenAndId]
	) -> UserProfileViewType? {
		guard photoCarouselEnabledUseCase.current, pictureIndex == Constants.Features.Timeline.photoCarouselIndex, picturesViewData.count > 1 else { return nil }
		return .photoCarousel(viewData: picturesViewData.map { $0.token })
	}

	internal func createUserProfileInfoViewDatas(
		profile: UserDomainModel,
		userMe: UserDomainModel
	) -> [PillViewData] {
		var viewDatas: [PillViewData] = []

		if let city = profile.cityOfResidence {
			let isTheSameCity = isMe ? false : city.city == userMe.cityOfResidence?.city

			let cityViewData = PillViewData(
				city: city.city,
				country: city.country,
				isTheSameCity: isTheSameCity
			)
			viewDatas.append(cityViewData)
		}

		if !isMe, let distance = profile.distance {
			let distanceStr = getDistanceString(for: distance, userIsMale: profile.isLocalizedMale)
			let distanceViewData = PillViewData(distance: distanceStr)
			viewDatas.append(distanceViewData)
		}

		if let jobViewData = PillViewData(
			job: profile.job,
			workplace: profile.workplace
		) {
			viewDatas.append(jobViewData)
		}

		if let school = profile.school, !school.isEmpty {
			let schoolViewData = PillViewData(school: school)
			viewDatas.append(schoolViewData)
		}

		if !romaBadgeUsecase.isEnabled,
		   let activityDate = profile.modificationDate,
		   let connectionVintageViewData = PillViewData(
			activityDate: activityDate,
			userMeIsMale: profile.isLocalizedMale
		   ) {
			viewDatas.append(connectionVintageViewData)
		}

		return viewDatas
	}

	internal func createUserTraitListViewDatas(
		traits: [TraitDomainModel],
		userMe: UserDomainModel,
		profile: UserDomainModel,
		showCommun: Bool = true
	) -> [PillViewData] {
		let commonAnswersTraitIDs = profile.commonTraitAndAnswers(with: userMe).compactMap { $0.id }

		var viewDatas = TraitAndAnswerViewData.getTraitsWithAnswersViewData(
			isUserMeMale: userMe.isLocalizedMale,
			traits: traits,
			traitsAnswers: profile.traits
		).compactMap { traitAndAnswer in
			let excludedTraitsIds = [Constants.Features.TraitMultiChoice.heightTraitId, Constants.Features.TraitMultiChoice.astrologyTraitId]
			let isCommonAnswer = commonAnswersTraitIDs.filter { !excludedTraitsIds.contains($0) }.contains(traitAndAnswer.trait.id) && showCommun
			return traitAndAnswer.toPillViewData(isCommonAnswer: isCommonAnswer)
		}

		if isLGBTEnabledUseCase.current,
		   let genderAlias = profile.genderAlias {
			viewDatas.insert(
				PillViewData(genderAlias: genderAlias.label),
				at: 0
			)
		}

		return viewDatas
	}

	internal func getDistanceString(
		for distance: Double,
		userIsMale: Bool
	) -> String {

		func getDistanceMetricString(distanceMetric: DistanceMetric) -> String {
			switch distanceMetric {
				case .feet(let distance):
					return L10n.profileDistanceFeet.applying(xibLocInfo: Str2StrXibLocInfo(pluralValue: NumberAndFormat(distance)))

				case .miles(let distance, let maximumFractionDigits):
					let numberFormatter = NumberFormatter()
					numberFormatter.numberStyle = .decimal
					numberFormatter.maximumFractionDigits = maximumFractionDigits
					return L10n.profileDistanceMiles.applying(xibLocInfo: Str2StrXibLocInfo(pluralValue: NumberAndFormat(distance, formatter: numberFormatter)))

				case .meters(let distance):
					return L10n.profileDistanceMeters.applying(xibLocInfo: Str2StrXibLocInfo(pluralValue: NumberAndFormat(distance)))

				case .kilometers(let distance, let maximumFractionDigits):
					let numberFormatter = NumberFormatter()
					numberFormatter.numberStyle = .decimal
					numberFormatter.maximumFractionDigits = maximumFractionDigits
					return L10n.profileDistanceKilometers.applying(xibLocInfo: Str2StrXibLocInfo(pluralValue: NumberAndFormat(distance, formatter: numberFormatter)))
			}
		}

		let distanceType = distance.stringFromDistanceOnProfile()
		var distanceStr = ""

		switch distanceType {
			case .lessThanBase(let metric):
				let metricStr = getDistanceMetricString(distanceMetric: metric)
				distanceStr = L10n.profileDistanceLessThan.applying(xibLocInfo: Str2StrXibLocInfo(replacement: metricStr))

			case .aboutBase(let metric):
				let metricStr = getDistanceMetricString(distanceMetric: metric)
				distanceStr = L10n.profileDistanceAbout.applying(xibLocInfo: Str2StrXibLocInfo(replacement: metricStr))

			case .moreThanBase(let metric):
				let metricStr = getDistanceMetricString(distanceMetric: metric)
				distanceStr = L10n.profileDistanceMoreThan.applying(xibLocInfo: Str2StrXibLocInfo(replacement: metricStr))

			case .none:
				break
		}

		return distanceStr.applying(xibLocInfo: Str2StrXibLocInfo(genderOtherIsMale: userIsMale))
	}

	internal func getSpotCategoryAvatarIconViewData(
		for category: SpotCategoryDomainModel
	) -> AvatarIconViewData {
		switch category {
			case .restaurant:
				return AvatarIconViewData(
					type: .icon(IconAsset.restaurant),
					showBorder: false,
					overridedTint: .mallow,
					overridedBackgroundColor: .fillToken(.default)
				)

			case .culture:
				return AvatarIconViewData(
					type: .icon(IconAsset.culture),
					showBorder: false,
					overridedTint: .skyDefault,
					overridedBackgroundColor: .fillToken(.default)
				)

			case .outdoor:
				return AvatarIconViewData(
					type: .icon(IconAsset.outdoor),
					showBorder: false,
					overridedTint: .mantis,
					overridedBackgroundColor: .fillToken(.default)
				)

			case .bar:
				return AvatarIconViewData(
					type: .icon(IconAsset.bar),
					showBorder: false,
					overridedTint: .butter,
					overridedBackgroundColor: .fillToken(.default)
				)

			case .club:
				return AvatarIconViewData(
					type: .icon(IconAsset.musicNote),
					showBorder: false,
					overridedTint: .cranberryDefault,
					overridedBackgroundColor: .fillToken(.default)
				)

			case .sport:
				return AvatarIconViewData(
					type: .icon(IconAsset.basketBall),
					showBorder: false,
					overridedTint: .coral,
					overridedBackgroundColor: .fillToken(.default)
				)
		}
	}

	internal func getCrossingType(
		userMe: UserDomainModel,
		profile: UserDomainModel,
		profileType: TimelineProfileRecommendationTypeDomainModel?,
		placeMark: CLPlacemark?
	) -> CardCrossingType? {
		guard let nbCrossing = profile.nbCrossings, profileType == .crossing || nbCrossing >= 1 else { return nil }
		if userMe.locationDisabled {
			return .locationDisable
		} else if let placemark = placeMark {
			return .location(placemark: placemark)
		} else {
			return .locationUnavailable
		}
	}

	internal func setupMapViewData(
		userMe: UserDomainModel,
		profile: UserDomainModel,
		profileType: TimelineProfileRecommendationTypeDomainModel?,
		placeMark: CLPlacemark?
	) -> CardActionViewData? {
		let crossingType = getCrossingType(userMe: userMe, profile: profile, profileType: profileType, placeMark: placeMark)

		let profileCrossingConfigurationEnabled = getProfileCrossingConfiguration.current.enabled

		guard let crossingType else { return nil }
		let titleType: CardCrossingTitleType
		if profileCrossingConfigurationEnabled,
		   let threshold = getProfileCrossingConfiguration.current.threshold {
			titleType = .withThreshold(threshold: threshold)
		} else {
			titleType = .standard
		}
		return CardActionViewData(
			type: crossingType,
			titleType: titleType,
			userName: profile.firstName ?? "",
			nbCrossing: profile.nbCrossings ?? 1,
			crossingDate: profile.lastMeetDate ?? Date(),
			userIsMale: userMe.isLocalizedMale,
			otherUserIsMale: profile.isLocalizedMale,
			userMePicture: userMe.picture?.url(),
			otherUserPicture: profile.picture?.url()
		)
	}

	private func createCommonTraitsData(
		traits: [TraitDomainModel],
		userMe: UserDomainModel,
		profile: UserDomainModel
	) -> [PillViewData] {
		// Height and Astrology information are excluded
		let excludedTraitsIds = [Constants.Features.TraitMultiChoice.heightTraitId, Constants.Features.TraitMultiChoice.astrologyTraitId]

		let commonAnswersTraitIDs = profile.commonTraitAndAnswers(with: userMe)
			.compactMap { $0.id }
			.filter { !excludedTraitsIds.contains($0) }

		let viewDatas = TraitAndAnswerViewData.getTraitsWithAnswersViewData(
			isUserMeMale: userMe.isLocalizedMale,
			traits: traits,
			traitsAnswers: profile.traits
		).filter { return commonAnswersTraitIDs.contains($0.trait.id) }.compactMap { return $0.toPillViewData() }

		return viewDatas
	}

	private func hobbiesAndSpots(
		userMe: UserDomainModel,
		profile: UserDomainModel
	) -> [PillViewData] {
		var result: [PillViewData] = []
		let hobbiesAnswersPillsViewData = UserTraitMultiChoiceViewData
			.commonMultiChoiceTraitAnswers(
				trait: getTraitsUseCase.current(trait: .hobbies),
				targetUser: profile,
				userMe: userMe
			)
			.map { PillViewData(text: $0) }

		var spotsViewData: [PillViewData] = []
		if let otherUserSpots = profile.spots {
			spotsViewData = otherUserSpots
				.filter { userMe.spots?.contains($0) ?? false }
				.map { PillViewData(text: $0.name) }
		}
		result = hobbiesAnswersPillsViewData + spotsViewData
		return result
	}
}
