//
//  ReactOnContentViewType.swift
//  happnTimeline
//
//  Created by Luc-Olivier Merson on 16/06/2025.
//

import SwiftUI

import happnPolis
import happnUI

public enum ReactOnContentStyle {
	case hug
	case fill
}

public struct ReactOnContentViewType: View {

	// MARK: - Private properties
	private let type: UserProfileViewType
	private let style: ReactOnContentStyle

	// MARK: - Initialization
	public init(
		type: UserProfileViewType,
		style: ReactOnContentStyle
	) {
		self.type = type
		self.style = style
	}

	// MARK: - Public properties
	public var body: some View {
		switch type {
			case let .relationshipUserInfosAndTraits(viewData):
				Container {
					CardMultiSelectionView(
						token: CardMultiSelectionToken(
							style: .white,
							pillSize: .medium,
							pillStyle: .colored(overridedInactiveBackgroundColor: .inverseWeak)
						),
						viewData: viewData
					)
				}

			case let .userInfos(viewData):
				UserInfosView(viewData: viewData)

			case let .photo(viewData, _):
				Container {
					CardUserView(
						token: CardUserToken(marge: .big),
						state: viewData
					)
				}
				.accessibilityIdentifier(id: "photo", children: .contain)

			case let .pictureContentCarousel(viewData):
				PictureContentTapCarouselView(
					items: viewData,
					token: PictureContentToken()
				)
				.accessibilityIdentifier(id: "picture_content_carousel", children: .contain)

			case let .trait(viewData):
				Container {
					CardPills(
						token: CardPillsToken(
							style: .white,
							pillSize: .medium,
							pillStyle: .colored(overridedInactiveBackgroundColor: .inverseWeak)
						),
						viewData: viewData
					)
				}
				.accessibilityIdentifier(id: "trait", children: .contain)

			case let .userInputAnswer(viewData):
				Container {
					CardPromptView(
						token: CardPromptToken(style: .white),
						viewData: viewData.promptViewData
					)
					.frame(maxWidth: .infinity)
				}
				.accessibilityIdentifier(id: "teasers", children: .contain)

			case let .traitMultiChoice(viewData):
				Container {
					CardPills(
						token: CardPillsToken(
							style: .white,
							pillSize: .large,
							pillStyle: .colored(overridedInactiveBackgroundColor: .inverseWeak)
						),
						viewData: viewData.cardPillsViewData
					)
				}
				.accessibilityIdentifier(id: "traitMultiChoice", children: .contain)

			case .communSpot(_):
				EmptyView() //Not Handled

			case let .spots(viewData):
				Container {
					CardPills(
						token: CardPillsToken(
							style: .white,
							pillSize: .large,
							pillStyle: .colored(overridedInactiveBackgroundColor: .inverseWeak)
						),
						viewData: viewData
					)
				}
				.accessibilityIdentifier(id: "spots", children: .contain)

			case let .description(viewData):
				Container {
					CardTextView(
						token: CardTextToken(
							type: .default,
							style: .white
						),
						viewData: viewData
					) {}
				}
				.accessibilityIdentifier(id: "description", children: .contain)

			case let .crossing(viewData):
				Container {
					CardActionView(
						token: CardActionToken(
							type: .default,
							smallButtonTopSpace: true
						),
						viewData: viewData,
						action: {}
					)
				} background: {
					CardActionBackgroundView(
						viewData: ImageViewComponentViewData(
							imageContentModeToken: .fillContainer,
							source: .image(IllustrationAsset.crossing.uiImage)
						),
						contentMode: .fill
					)
				}
				.accessibilityIdentifier(id: "crossing", children: .contain)

			case let .blockReport(viewData):
				ButtonView(
					token: ButtonToken(
						level: .secondary,
						theme: .dark,
						size: .small,
						constraintMode: .hug
					),
					viewData: viewData,
					action: {}
				)
				.accessibilityIdentifier(id: "blockAndReportButton")

			case let .spacer(token):
				Spacer()
					.frame(height: token.size)
		}
	}

	// MARK: - Private methods
	private func Container(
		content: () -> some View,
		background: () -> some View = { EmptyView() }
	) -> some View {
		switch style {
			case .hug:
				let token = CardContainerToken(
					style: .white,
					constraintMode: .hug
				)
				return AnyView(
					CardContainerView(
						padding: type.padding,
						token: token,
						content: content,
						background: background
					)
				)

			case .fill:
				let token = CardContainerToken(
					style: .white,
					constraintMode: .fill
				)
				return AnyView(
					CardFillContainerView(
						padding: type.padding,
						token: token
					) {
						content()
					} background: {
						background()
					}
				)
		}
	}
}

extension UserProfileViewType {
	public var padding: EdgeInsets {
		switch self {
			case .photo:
				return .zero

			default:
				return EdgeInsets(SpacingToken.spacing32.size)
		}
	}
}
