//
//  CardActionViewData+Extension.swift
//
//
//  Created by Sunggweon Hyeong on 27/09/2024.
//

import Foundation
import MapKit

import XibLoc

import happnPolis
import happnUtils

private extension CardCrossingType {
	var icon: IconAsset {
		switch self {
			case .location:
				return IconAsset.location

			case .locationDisable:
				return IconAsset.locationTick

			case .locationUnavailable:
				return IconAsset.locationTick
		}
	}

	var overridedTint: PolisColor.IconToken {
		switch self {
			case .location:
				return .inverseWeaker

			case .locationDisable:
				return .default

			case .locationUnavailable:
				return .inverseWeaker
		}
	}

	var overridedTextColor: PolisColor.TextToken {
		switch self {
			case .location:
				return .default

			case .locationDisable:
				return .inverseWeaker

			case .locationUnavailable:
				return .default
		}
	}

	var overridedBackgroundColor: PolisColor.FillToken {
		switch self {
			case .location:
				return .lavender

			case .locationDisable:
				return .default

			case .locationUnavailable:
				return .inverseWeak
		}
	}

	var overridedAvatarBackgroundColor: PolisColor.FillToken {
		switch self {
			case .location:
				return .default

			case .locationDisable:
				return .inverseWeak

			case .locationUnavailable:
				return .default
		}
	}

	var buttonText: String? {
		switch self {
			case let .location(placemark):
				guard let crossingAddress = placemark.crossingAddressString(), !crossingAddress.isEmpty else { return nil }
				return crossingAddress

			case .locationDisable:
				return L10n.profileViewCrossingsViewerHideLocationButton

			case .locationUnavailable:
				return L10n.profileViewCrossingsVieweeHideLocation
		}
	}

	func dateText(crossingDate: Date?) -> String {
		var dateText = Language.string(fromDate: crossingDate ?? Date())
		guard let crossingAddress = placemark?.crossingAddressString(), !crossingAddress.isEmpty else { return dateText }
		dateText += " " + L10n.profileCrossingsNear
		return dateText
	}
}

extension CardActionViewData {
	init(
		type: CardCrossingType,
		titleType: CardCrossingTitleType,
		userName: String,
		nbCrossing: Int,
		crossingDate: Date?,
		userIsMale: Bool,
		otherUserIsMale: Bool,
		userMePicture: URL?,
		otherUserPicture: URL?
	) {
		let title = switch titleType {
			case .standard:
				L10n.profileCrossingsMapNumberOfCrossings
					.applying(
						xibLocInfo: Str2StrXibLocInfo(
							replacement: userName,
							pluralValue: NumberAndFormat(max(1, nbCrossing)),
							genderMeIsMale: userIsMale,
							genderOtherIsMale: otherUserIsMale
						)
					)

			case .withThreshold(let threshold):
				if nbCrossing <= threshold {
					L10n.profileCrossingsMapNumberOfCrossings
						.applying(
							xibLocInfo: Str2StrXibLocInfo(
								replacement: userName,
								pluralValue: NumberAndFormat(max(1, nbCrossing)),
								genderMeIsMale: userIsMale,
								genderOtherIsMale: otherUserIsMale
							)
						)
				} else {
					L10n.profileCrossingsMapSeveralCrossings.applying(xibLocInfo: Str2StrXibLocInfo(replacement: userName))
				}
		}

		var avatarViewDatas: [AvatarIconViewData] = []

		if let userMePicture {
			let userMeAvatarIconViewData = AvatarIconViewData(type: .picture(.remote(userMePicture)))
			avatarViewDatas.append(userMeAvatarIconViewData)
		}
		if let otherUserPicture {
			let otherUserAvatarIconViewData = AvatarIconViewData(type: .picture(.remote(otherUserPicture)))
			avatarViewDatas.append(otherUserAvatarIconViewData)
		}

		self.init(
			avatarSetViewData: AvatarSetViewData(avatarViewDatas),
			buttonViewData: ButtonViewData(
				title: type.buttonText,
				avatarIconViewData: AvatarIconViewData(
					type: .icon(type.icon),
					overridedTint: type.overridedTint,
					overridedBackgroundColor: .fillToken(type.overridedAvatarBackgroundColor)
				),
				overridedTextColor: type.overridedTextColor,
				overridedBackgroundColor: type.overridedBackgroundColor
			),
			imageViewData: ImageViewComponentViewData(
				imageContentModeToken: .fitContainer,
				source: .image(Asset.FeaturesAssets.cardActionMapPattern.image)
			),
			backgroundColorToken: .lavender,
			showBorder: true,
			titleText: title,
			descriptionText: type.dateText(crossingDate: crossingDate),
			overrideDescriptionFont: PolisFont.Body.medium,
			buttonShadow: .dark2
		)
	}
}
