//
//  UserProfileCTAViewData.swift
//
//
//  Created by Nicolas Sabella on 11/12/2024.
//

import happnPolis
import happnUI

public enum BoostState {
	case `default`
	case running
	case available
}

public enum UserProfileViewInputType: Hashable {
	case button(type: UserProfileViewButton)
	case spacer
}

public enum UserProfileViewSpacer: Hashable {
	case spacer
}

public enum UserProfileViewButton: Hashable {
	case boost(state: BoostState)
	case like(isPrioritaryLikeActivated: Bool)
	case reject(shouldBeLarge: Bool, overridedTintColor: PolisColor.IconToken? = .butter)
	case superCrush(isSuperCrushCreditRemaining: Bool)

	public var accessibilityIdentifier: String {
		switch self {
			case let .boost(state):
				switch state {
					case .default, .available:
						"boostButton"

					case .running:
						"boostButtonActive"
				}

			case .like:
				"likeButton"

			case .reject:
				"rejectButton"

			case .superCrush:
				"superCrushButton"
		}
	}

	public var shouldDisabledButtons: Bool {
		switch self {
			case .like, .reject:
				return true

			default:
				return false
		}
	}

	public var icon: IconAssetToken? {
		switch self {
			case let .boost(state):
				switch state {
					case .default, .available:
						IconAsset.boost

					case .running:
						nil
				}

			case .like:
				IconAsset.heart

			case .reject:
				IconAsset.close

			case .superCrush:
				IconAsset.supercrush
		}
	}

	public var lottieViewData: LottieComponentViewData? {
		switch self {
			case .boost(.running):
				return LottieComponentViewData(
					type: .json(.boostActive),
					loopMode: .loop,
					backgroundBehavior: .pauseAndRestore
				)

			default:
				return nil
		}
	}

	public var overridedTintColor: PolisColor.IconToken? {
		switch self {
			case let .boost(state):
				switch state {
					case .default:
							.coral

					case .running, .available:
							.default
				}

			case let .like(isPrioritaryLikeActivated):
				isPrioritaryLikeActivated ? .default : .cranberryDefault

			case let .superCrush(isSuperCrushCreditRemaining):
				isSuperCrushCreditRemaining ? .default : .skyDefault

			case let .reject(_, overridedTintColor):
				overridedTintColor
		}
	}

	public var overridedBackgroundColor: PolisColor.FillToken? {
		switch self {
			case let .boost(state):
				switch state {
					case .default:
							.default

					case .running, .available:
							.coral
				}

			case let .like(isPrioritaryLikeActivated):
				isPrioritaryLikeActivated ? .cranberry : .default

			case let .superCrush(isSuperCrushCreditRemaining):
				isSuperCrushCreditRemaining ? .sky : .default

			default:
				nil
		}
	}

	public var buttonToken: ButtonToken {
		switch self {
			case .boost:
				ButtonToken(
					level: .primary,
					theme: .dark,
					size: .medium,
					constraintMode: .hug,
					hapticType: .impact(.soft)
				)

			case .like:
				ButtonToken(
					level: .primary,
					theme: .dark,
					size: .large,
					constraintMode: .hug,
					hapticType: .impact(.soft)
				)

			case let .reject(shouldBeLarge, _):
				ButtonToken(
					level: .primary,
					theme: .dark,
					size: shouldBeLarge ? .large : .medium,
					constraintMode: .hug
				)

			case .superCrush:
				ButtonToken(
					level: .primary,
					theme: .dark,
					size: .medium,
					constraintMode: .hug,
					hapticType: .impact(.medium)
				)
		}
	}
}

public struct UserProfileCTAViewData: Equatable {

	public let activeButtonTypes: [UserProfileViewInputType]
	public let shouldHaveBackground: Bool

	public init(
		activeButtonTypes: [UserProfileViewInputType],
		shouldHaveBackground: Bool? = nil
	) {
		self.activeButtonTypes = activeButtonTypes
		self.shouldHaveBackground = shouldHaveBackground ?? !activeButtonTypes.isEmpty
	}
}
