//
//  RewindAfterPurchaseWorker.swift
//  happnTimeline
//
//  Created by Luc-Olivier Merson on 14/11/2025.
//

import Combine
import Foundation

import happnConfiguration
import happnNavigation
import happnShops
import happnTracking
import happnUsers
import happnUtils

public protocol RewindAfterPurchaseWorkerProtocol: WorkerProtocol {}

class RewindAfterPurchaseWorker: RewindAfterPurchaseWorkerProtocol, HasCancellables {

	// MARK: - Private properties
	private let rewindUseCase: ProfileDisplayRewindUseCaseProtocol
	private let lastShopOriginStateRepository: LastShopOriginStateRepositoryProtocol
	private let rewindEligibilityUseCase: ProfileDisplayRewindEligibilityUseCaseProtocol
	private let orderCompletedPopupClosedStateRepository: OrderCompletedPopupClosedStateRepositoryProtocol

	@Published var canRewind = false

	// MARK: - Initialisation
	public init(
		rewindUseCase: ProfileDisplayRewindUseCaseProtocol,
		lastShopOriginStateRepository: LastShopOriginStateRepositoryProtocol,
		rewindEligibilityUseCase: ProfileDisplayRewindEligibilityUseCaseProtocol,
		orderCompletedPopupClosedStateRepository: OrderCompletedPopupClosedStateRepositoryProtocol
	) {
		self.rewindUseCase = rewindUseCase
		self.lastShopOriginStateRepository = lastShopOriginStateRepository
		self.rewindEligibilityUseCase = rewindEligibilityUseCase
		self.orderCompletedPopupClosedStateRepository = orderCompletedPopupClosedStateRepository
	}

	// MARK: - Public methods
	func start() {
		rewindEligibilityUseCase.observe().publisher
			.ignoreError
			.eraseToAnyPublisher()
			.receive(on: DispatchQueue.main)
			.map { $0 == .canRewind }
			.weakAssign(to: \.canRewind, on: self)
			.store(in: &cancellables)

		orderCompletedPopupClosedStateRepository.publisher
			.receive(on: DispatchQueue.main)
			.sink { [weak self] _ in
				guard let self, let origin = lastShopOriginStateRepository.current, canRewind, origin.isFromRewind else { return }
				rewindUseCase.rewind(screen: origin.rewindScreen ?? .timeline).publisher
					.receive(on: DispatchQueue.main)
					.sink()
					.store(in: &cancellables)
			}.store(in: &cancellables)
	}
}

extension ShopOrigin {
	var rewindScreen: ScreenName? {
		switch self {
			case .rewind:
				return .timeline

			case .rewindMissedCrush:
				return .missedCrush

			default:
				return nil
		}
	}
}
