//
//  UserProfilesViewUseCase.swift
//
//
//  Created by Sunggweon Hyeong on 27/09/2024.
//

import Combine
import CoreLocation
import Foundation
import MapKit

import RxCombine
import XibLoc

import happnConfiguration
import happnConstants
import happnFlashnote
import happnLogger
import happnMap
import happnPolis
import happnShops
import happnSpots
import happnTraits
import happnUsers
import happnUtils
import happnViewDataUtils

public struct UserProfilesUseCasesContainer {
	public let badgeUseCase: ProfileDisplayUserBadgeUseCaseProtocol
	public let mapCacheManager: MapCacheManagerProtocol
	public let userMeUseCase: GetUserMeUseCaseProtocol
	public let actionButtonUseCase: ActionButtonUseCaseProtocol
	public let isLGBTEnabledUseCase: IsLGBTEnabledUseCaseProtocol
	public let getSpotConfigurationUseCase: GetSpotConfigurationUseCaseProtocol
	public let getTraitsUseCase: GetTraitsUseCaseProtocol
	public let romaBadgeUsecase: RomaV2UseCaseProtocol
	public let isReplyToTeasersEnabledUseCase: IsReplyToTeasersEnabledUseCaseProtocol
	public let getProfileCrossingConfiguration: GetProfileCrossingConfigurationUseCaseProtocol
	public let romaV2UseCase: RomaV2UseCaseProtocol
	public let photoCarouselEnabledUseCase: PhotoCarouselEnabledUseCaseProtocol

	public init(
		badgeUseCase: ProfileDisplayUserBadgeUseCaseProtocol,
		mapCacheManager: MapCacheManagerProtocol,
		userMeUseCase: GetUserMeUseCaseProtocol,
		actionButtonUseCase: ActionButtonUseCaseProtocol,
		isLGBTEnabledUseCase: IsLGBTEnabledUseCaseProtocol,
		getSpotConfigurationUseCase: GetSpotConfigurationUseCaseProtocol,
		getTraitsUseCase: GetTraitsUseCaseProtocol,
		romaBadgeUsecase: RomaV2UseCaseProtocol,
		isReplyToTeasersEnabledUseCase: IsReplyToTeasersEnabledUseCaseProtocol,
		getProfileCrossingConfiguration: GetProfileCrossingConfigurationUseCaseProtocol,
		romaV2UseCase: RomaV2UseCaseProtocol,
		photoCarouselEnabledUseCase: PhotoCarouselEnabledUseCaseProtocol
	) {
		self.badgeUseCase = badgeUseCase
		self.mapCacheManager = mapCacheManager
		self.userMeUseCase = userMeUseCase
		self.actionButtonUseCase = actionButtonUseCase
		self.isLGBTEnabledUseCase = isLGBTEnabledUseCase
		self.getSpotConfigurationUseCase = getSpotConfigurationUseCase
		self.getTraitsUseCase = getTraitsUseCase
		self.romaBadgeUsecase = romaBadgeUsecase
		self.isReplyToTeasersEnabledUseCase = isReplyToTeasersEnabledUseCase
		self.getProfileCrossingConfiguration = getProfileCrossingConfiguration
		self.romaV2UseCase = romaV2UseCase
		self.photoCarouselEnabledUseCase = photoCarouselEnabledUseCase
	}
}

public enum OpenProfileOrigin: Equatable {
	case conversation
	case crush
	case deeplink
	case listOfLikes
	case map
	case spot
	case superCrush
	case notification
	case none
}

public enum UserProfilesType: Equatable {
	case flashnote
	case vibes
	case timeline
	case openProfile(origin: OpenProfileOrigin)

	public var isStack: Bool {
		switch self {
			case .vibes, .timeline:
				return true

			case .flashnote, .openProfile:
				return false
		}
	}

	public var fromDeepLink: Bool {
		switch self {
			case let .openProfile(origin):
				return origin == .deeplink

			case .flashnote, .vibes, .timeline:
				return false
		}
	}

	public var isOpenProfileFromConversation: Bool {
		switch self {
			case let .openProfile(origin):
				return origin == .conversation

			case .flashnote, .vibes, .timeline:
				return false
		}
	}

	public var pinchToZoomCoordinateSpace: String? {
		switch self {
			case .openProfile:
				return Constants.Features.OpenProfile.coordinateSpace

			default:
				return nil
		}
	}

	public var actionButtonsHorizontalSpacing: SpacingToken {
		switch self {
			case .timeline:
				return SpacingToken.spacing20

			case .openProfile, .vibes, .flashnote:
				return SpacingToken.spacing40
		}
	}
}

public protocol UserProfilesViewUseCaseProtocol {
	var publisher: AnyPublisher<[UserProfileViewType], Never> { get }
	var viewData: [UserProfileViewType] { get }
	var commonPointsCount: Int { get }
}

public class UserProfilesViewUseCase: UserProfilesViewUseCaseProtocol {
	private let badgeUseCase: ProfileDisplayUserBadgeUseCaseProtocol
	private let mapCacheManager: MapCacheManagerProtocol
	private let userMeUseCase: GetUserMeUseCaseProtocol
	private let actionButtonUseCase: ActionButtonUseCaseProtocol
	private let isReplyToTeasersEnabledUseCase: IsReplyToTeasersEnabledUseCaseProtocol
	private let getSpotConfigurationUseCase: GetSpotConfigurationUseCaseProtocol
	private let profileType: TimelineProfileRecommendationTypeDomainModel?

	let photoCarouselEnabledUseCase: PhotoCarouselEnabledUseCaseProtocol
	let romaBadgeUsecase: RomaV2UseCaseProtocol
	let isLGBTEnabledUseCase: IsLGBTEnabledUseCaseProtocol
	let getTraitsUseCase: GetTraitsUseCaseProtocol
	let getProfileCrossingConfiguration: GetProfileCrossingConfigurationUseCaseProtocol
	let profile: UserDomainModel
	let flashnote: FlashnoteDomainModel?
	let type: UserProfilesType

	public var publisher: AnyPublisher<[UserProfileViewType], Never> {
		subject.eraseToAnyPublisher()
	}

	private var subject: PassthroughSubject<[UserProfileViewType], Never> = PassthroughSubject()
	private var cancellables = Set<AnyCancellable>()

	public var viewData: [UserProfileViewType] = []
	public var commonPointsCount: Int = 0

	internal var isMe: Bool {
		return userMeUseCase.current().id == profile.id
	}

	var isProfileSponsored: Bool {
		return profile.type == .sponsor
	}

	public init(
		userProfilesUseCasesContainer: UserProfilesUseCasesContainer,
		profileType: TimelineProfileRecommendationTypeDomainModel?,
		profile: UserDomainModel,
		flashnote: FlashnoteDomainModel? = nil,
		type: UserProfilesType
	) {

		self.badgeUseCase = userProfilesUseCasesContainer.badgeUseCase
		self.mapCacheManager = userProfilesUseCasesContainer.mapCacheManager
		self.userMeUseCase = userProfilesUseCasesContainer.userMeUseCase
		self.actionButtonUseCase = userProfilesUseCasesContainer.actionButtonUseCase
		self.isLGBTEnabledUseCase = userProfilesUseCasesContainer.isLGBTEnabledUseCase
		self.getSpotConfigurationUseCase = userProfilesUseCasesContainer.getSpotConfigurationUseCase
		self.getTraitsUseCase = userProfilesUseCasesContainer.getTraitsUseCase
		self.isReplyToTeasersEnabledUseCase = userProfilesUseCasesContainer.isReplyToTeasersEnabledUseCase
		self.getProfileCrossingConfiguration = userProfilesUseCasesContainer.getProfileCrossingConfiguration
		self.photoCarouselEnabledUseCase = userProfilesUseCasesContainer.photoCarouselEnabledUseCase
		self.romaBadgeUsecase = userProfilesUseCasesContainer.romaBadgeUsecase
		self.profileType = profileType
		self.profile = profile
		self.flashnote = flashnote
		self.type = type

		let currentUserMe = userMeUseCase.current()
		self.viewData = createUserProfileViewData(
			userMe: currentUserMe,
			badgeType: badgeUseCase.getSyncProfileBadge(for: profile, userMe: currentUserMe),
			placemark: nil
		)

		createUserProfilesViewData()
	}

	// MARK: - Private
	private func createUserProfilesViewData() {
		let userMePublisher = userMeUseCase.observe().publisher.ignoreError

		let badgePublisher = badgeUseCase
			.getProfileBadge(for: profile, userMe: userMeUseCase.current())
			.publisher
			.replaceError(with: nil)

		let locationPublisher = mapCacheManager
			.getReverseGeocodeLocation(profile.lastMetPosition?.getLocation())
			.publisher
			.prepend(nil)
			.replaceError(with: nil)

		Publishers.CombineLatest3(
			userMePublisher,
			badgePublisher,
			locationPublisher
		)
		.receive(on: DispatchQueue.main)
		.compactMap { [weak self] userMe, badgeType, placemark in
			return self?.createUserProfileViewData(
				userMe: userMe,
				badgeType: badgeType,
				placemark: placemark
			)
		}
		.subscribe(subject)
		.store(in: &cancellables)
	}

	private func isReplyToTeasersEnabled() -> Bool {
		return type != .flashnote && isReplyToTeasersEnabledUseCase.current
	}

	private func createUserProfileViewData(
		userMe: UserDomainModel,
		badgeType: UserBadgeType?,
		placemark: CLPlacemark?
	) -> [UserProfileViewType] {
		let photosViewData = createPhotosViewData(
			type: type,
			badgeType: badgeType,
			flashnote: flashnote,
			userMe: userMe,
			profile: profile
		)
		let pictureContentCarouselViewData = createPictureContentCarouselViewData(
			type: type,
			badgeType: badgeType,
			flashnote: flashnote,
			userMe: userMe,
			profile: profile
		)
		let traits = getTraitsUseCase.current() ?? []
		let isReplyToTeasersEnabled = isReplyToTeasersEnabled()

		let profileItems: [UserProfileViewType] = [
			createUserInfoViewData(
				getTraitsUseCase: getTraitsUseCase,
				type: type,
				badgeType: badgeType,
				flashnote: flashnote,
				userMe: userMe,
				profile: profile
			),
			addPictureContentCarouselIfAvailable(
				viewData: pictureContentCarouselViewData
			),
			addPictureIfAvailable(
				pictureIndex: 0,
				picturesViewData: photosViewData
			),
			getProfileCrossingConfiguration.current.enabled ? createMapViewData(
				type: type,
				userMe: userMe,
				profile: profile,
				profileType: profileType,
				placemark: placemark
			) : nil,
			createRelationshipUserInfosAndTraitsViewData(
				type: type,
				relationshipTrait: getTraitsUseCase.current(trait: .relationship),
				traits: traits,
				userMe: userMe,
				profile: profile
			),
			createUserInputAnswerViewData(
				type: type,
				userMe: userMe,
				profile: profile,
				traits: traits,
				index: 0,
				isReplyToTeasersEnabled: isReplyToTeasersEnabled
			),
			addPictureIfAvailable(
				pictureIndex: 1,
				picturesViewData: photosViewData
			),
			createHobbiesViewData(
				type: type,
				traitUseCase: getTraitsUseCase,
				userMe: userMe,
				profile: profile
			),
			createUserInputAnswerViewData(
				type: type,
				userMe: userMe,
				profile: profile,
				traits: traits,
				index: 1,
				isReplyToTeasersEnabled: isReplyToTeasersEnabled
			),
			addPictureIfAvailable(
				pictureIndex: 2,
				picturesViewData: photosViewData
			),
			createSpotsViewData(
				type: type,
				spotsIsEnabled: getSpotConfigurationUseCase.isEnabled,
				userMe: userMe,
				profile: profile
			),
			createDescriptionViewData(
				type: type,
				profile: profile
			),
			addPictureIfAvailable(
				pictureIndex: 3,
				picturesViewData: photosViewData
			),
			addPictureIfAvailable(
				pictureIndex: 4,
				picturesViewData: photosViewData
			),
			createUserInputAnswerViewData(
				type: type,
				userMe: userMe,
				profile: profile,
				traits: traits,
				index: 2,
				isReplyToTeasersEnabled: isReplyToTeasersEnabled
			),
			addPictureIfAvailable(
				pictureIndex: 5,
				picturesViewData: photosViewData
			),
			!getProfileCrossingConfiguration.current.enabled ? createMapViewData(
				type: type,
				userMe: userMe,
				profile: profile,
				profileType: profileType,
				placemark: placemark
			) : nil,
			createBlockReportViewData(
				type: type,
				profile: profile
			)
		].compactMap { $0 }

		return profileItems
	}
}
