import Foundation
import Nimble
import Quick
import RxSwift
import XCTest

import happnUsers
import happnUsersMock

@testable import happnProfileUtils

class ActionButtonUseCaseTests: QuickSpec {
	override class func spec() {
		var userMe: UserDomainModel!
		var actionButtonUseCase: ActionButtonUseCaseProtocol!
		var disposeBag: DisposeBag!

		beforeEach {
			userMe = UserDomainModel.mock(id: "me")
			actionButtonUseCase = ActionButtonUseCase()
			disposeBag = DisposeBag()
		}

		it("Should return me") {
			let user = UserDomainModel.mock(id: "me")

			waitUntil { done in
				actionButtonUseCase.actionButtonType(for: user, userMe: userMe).subscribe(onSuccess: {
					let type = $0
					expect(type).toNot(beNil())
					expect({
						guard case ActionButtonConfigurationType.me = type! else { return false }
						return true
					}).to(beTrue())
					done()
				}).disposed(by: disposeBag)
			}
		}

		it("Should return charmedMe") {
			let user = UserDomainModel.mock(
				relationToMe: nil,
				hasCharmedMe: true,
				hasAcceptedMe: true,
				isCharmed: true
			)

			waitUntil { done in
				actionButtonUseCase.actionButtonType(for: user, userMe: userMe).subscribe(onSuccess: {
					let type = $0
					expect(type).toNot(beNil())
					expect({
						guard case ActionButtonConfigurationType.charmedMe = type! else { return false }
						return true
					}).to(beTrue())
					done()
				}).disposed(by: disposeBag)
			}
		}

		it("Should return crushed above charmedMe") {
			let user = UserDomainModel.mock(relationToMe: .mutual(date: nil),
											hasCharmedMe: true,
											hasAcceptedMe: true,
											isCharmed: true)

			waitUntil { done in
				actionButtonUseCase.actionButtonType(for: user, userMe: userMe).subscribe(onSuccess: {
					let type = $0
					expect(type).toNot(beNil())
					expect({
						guard case ActionButtonConfigurationType.crushed = type! else { return false }
						return true
					}).to(beTrue())
					done()
				}).disposed(by: disposeBag)
			}
		}

		it("Should return alreadyCharmed above alreadyReacted") {
			let user = UserDomainModel.mock(relationToMe: .accepted(date: nil),
											hasCharmedMe: false,
											hasAcceptedMe: false,
											isCharmed: true)

			waitUntil { done in
				actionButtonUseCase.actionButtonType(for: user, userMe: userMe).subscribe(onSuccess: {
					let type = $0
					expect(type).toNot(beNil())
					expect({
						guard case ActionButtonConfigurationType.alreadyCharmed = type! else { return false }
						return true
					}).to(beTrue())
					done()
				}).disposed(by: disposeBag)
			}
		}
	}
}
