import CoreLocation
import Foundation
import happnUsers
import RxSwift

public protocol ActionButtonUseCaseProtocol {
	func actionButtonType(for user: UserDomainModel, userMe: UserDomainModel) -> Single<ActionButtonConfigurationType?>
	func getActionButtonType(for user: UserDomainModel, userMe: UserDomainModel) -> ActionButtonConfigurationType?
	func getActionButtonForAds() -> Single<ActionButtonConfigurationType?>
	func didUserLikeMe(for user: UserDomainModel, userMe: UserDomainModel) -> Single<Bool>
}

public class ActionButtonUseCase: ActionButtonUseCaseProtocol {

	public init() {}

	public func getActionButtonForAds() -> Single<ActionButtonConfigurationType?> {
		return Single.just(.ads)
	}

	public func getActionButtonType(for user: UserDomainModel, userMe: UserDomainModel) -> ActionButtonConfigurationType? {
		return calculateRelationship(for: user, userMe: userMe)
	}

	public func actionButtonType(for user: UserDomainModel, userMe: UserDomainModel) -> Single<ActionButtonConfigurationType?> {
		return Single.create { [weak self] single -> Disposable in
			single(.success(self?.calculateRelationship(for: user, userMe: userMe)))
			return Disposables.create()
		}
	}

	public func didUserLikeMe(for user: UserDomainModel, userMe: UserDomainModel) -> Single<Bool> {
		return Single.create { [weak self] single -> Disposable in
			let calculateRelationship = self?.calculateRelationship(for: user, userMe: userMe)
			if calculateRelationship == .likedMe {
				single(.success(true))
			} else {
				single(.success(false))
			}
			return Disposables.create()
		}
	}

	private func calculateRelationship(for user: UserDomainModel, userMe: UserDomainModel?) -> ActionButtonConfigurationType? {
		if user.id == userMe?.id {
			return .me
		}

		switch (
			user.isCharmed,
			user.hasCharmedMe,
			user.hasAcceptedMe,
			user.relationToMe == .some(.mutual(date: nil)),
			userMe?.subscriptionLevel.canAccessToLoL,
			user.type == .sponsor
		) {
			case (    _, _, _, _, _, true):
				if user.relationToMe?.isMutual ?? false {
					return .sponsoredAlreadyReacted
				} else {
					return .sponsored
				}

			case (    _, true, _, false, _, _):
				return .charmedMe

			case (    _, _, true, false, true, _):
				return .likedMe

			case (    _, _, _, true, _, _):
				return .crushed

			case ( true, _, _, _, _, _):
				return .alreadyCharmed

			default:
				break
		}

		switch user.relationToMe {
			case .accepted(date: _):
				return .alreadyReacted

			default:
				break
		}

		return nil
	}
}
