//
//  UserInfosViewData.swift
//
//
//  Created by Nicolas Sabella on 09/07/2025.
//

import Foundation

import happnConfiguration
import happnPolis
import happnUsers
import happnUtils
import XibLoc

public struct UserInfosViewData: Equatable {
	public let nameAndAgeViewData: LabelWithIconViewData
	public let crossingInfosViewData: LabelWithIconViewData?
	public let activityInfosViewData: LabelWithIconViewData?
	public let relationshipInfosViewData: LabelWithIconViewData?
	public let relationShipInfosTextColor: PolisColor.TextToken?
	public let relationShipInfosTintColor: PolisColor.IconToken?
	public let bannerViewData: BannerViewData?
	public let expandableBannerViewData: ExpandableBannerViewData?

	public init(
		otherUser: UserDomainModel,
		userMe: UserDomainModel,
		superCrushMessage: String?,
		banner: BannerViewData? = nil,
		expandableBanner: ExpandableBannerViewData? = nil,
		badgeType: UserBadgeType?,
		romaV2Enabled: Bool,
		isFromConversation: Bool = false
	) {
		var avatarSetIconViewTypes: [AvatarIconViewType] = []
		if let userMeURL = userMe.mainPicture?.url(),
		   let otherUserMeURL = otherUser.mainPicture?.url() {
			avatarSetIconViewTypes = [
				.picture(.remote(userMeURL)),
				.picture(.remote(otherUserMeURL))
			]
		}

		nameAndAgeViewData = LabelWithIconViewData(
			nameAndAge: otherUser.nameAndAge,
			isCertified: otherUser.isCertified
		)

		if romaV2Enabled {
			if let badgeType = badgeType {
				switch badgeType {
					case .likedMe:
						relationshipInfosViewData = LabelWithIconViewData(
							text: L10n.reactionReceivedText2.applying(
								xibLocInfo: Str2StrXibLocInfo(
									genderMeIsMale: userMe.isLocalizedMale,
									genderOtherIsMale: otherUser.isLocalizedMale
								)
							),
							numberOfLines: 1,
							leftIcon: IconAsset.heart
						)
						relationShipInfosTextColor = .cranberry
						relationShipInfosTintColor = .cranberryBolder
						expandableBannerViewData = expandableBanner
						bannerViewData = nil

					case .charmedMe:
						relationshipInfosViewData = LabelWithIconViewData(
							text: L10n.profileViewSupercrushReceived.applying(
								xibLocInfo: Str2StrXibLocInfo(
									genderMeIsMale: userMe.isLocalizedMale,
									genderOtherIsMale: otherUser.isLocalizedMale
								)
							),
							numberOfLines: 1,
							leftIcon: IconAsset.supercrush
						)
						relationShipInfosTextColor = .sky
						relationShipInfosTintColor = .skyBolder

						if let superCrushMessage = superCrushMessage {
							expandableBannerViewData = nil
							bannerViewData = BannerViewData(
								avatarIconViewData: AvatarIconViewData(
									type: .icon(IconAsset.supercrush),
									overridedTint: .skyDefault,
									overridedBackgroundColor: .backgroundToken(.inverse)
								),
								message: superCrushMessage
							)
						} else {
							expandableBannerViewData = expandableBanner
							bannerViewData = nil
						}

					case .crushed:
						relationshipInfosViewData = LabelWithIconViewData(
							text: L10n.reactionAlreadyCrushText.applying(
								xibLocInfo: Str2StrXibLocInfo(
									genderMeIsMale: userMe.isLocalizedMale,
									genderOtherIsMale: otherUser.isLocalizedMale
								)
							),
							numberOfLines: 1,
							leftIcon: IconAsset.lovely
						)
						relationShipInfosTextColor = .mallow
						relationShipInfosTintColor = .mallow
						expandableBannerViewData = expandableBanner
						bannerViewData = nil

					case .alreadyCharmed:
						relationshipInfosViewData = LabelWithIconViewData(
							text: L10n.profileViewSupercrushSent.applying(
								xibLocInfo: Str2StrXibLocInfo(
									genderMeIsMale: userMe.isLocalizedMale,
									genderOtherIsMale: otherUser.isLocalizedMale
								)
							),
							numberOfLines: 1,
							leftIcon: IconAsset.supercrush
						)
						relationShipInfosTextColor = .sky
						relationShipInfosTintColor = .skyBolder
						expandableBannerViewData = expandableBanner
						bannerViewData = nil

					case .alreadyReacted:
						relationshipInfosViewData = LabelWithIconViewData(
							text: L10n.profileViewLikeSent.applying(
								xibLocInfo: Str2StrXibLocInfo(
									genderMeIsMale: userMe.isLocalizedMale,
									genderOtherIsMale: otherUser.isLocalizedMale
								)
							),
							numberOfLines: 1,
							leftIcon: IconAsset.heart
						)
						relationShipInfosTextColor = .cranberry
						relationShipInfosTintColor = .cranberryBolder
						expandableBannerViewData = expandableBanner
						bannerViewData = nil

					case .comet:
						bannerViewData = BannerViewData(
							avatarSetViewData: AvatarSetViewData(
								avatarSetIconViewTypes.map { icon in
									AvatarIconViewData(
										type: icon,
										showBorder: true
									)
								}
							),
							title: .text(L10n.badgeRomaCompatibilityTitle),
							extendFrame: false,
							showAvatarSetLottie: true
						)
						relationshipInfosViewData = nil
						relationShipInfosTextColor = nil
						relationShipInfosTintColor = nil
						expandableBannerViewData = nil

					case .cometExplanation:
						bannerViewData = BannerViewData(
							avatarSetViewData: AvatarSetViewData(
								avatarSetIconViewTypes.map { icon in
									AvatarIconViewData(
										type: icon,
										showBorder: true
									)
								}
							),
							title: .text(L10n.badgeRomaCompatibilityTitle),
							message: otherUser.cometExplanation,
							extendFrame: true,
							animateText: true,
							showAvatarSetLottie: true
						)
						relationshipInfosViewData = nil
						relationShipInfosTextColor = nil
						relationShipInfosTintColor = nil
						expandableBannerViewData = nil

                case .sponsored:
                    relationshipInfosViewData = nil
                    relationShipInfosTextColor = nil
                    relationShipInfosTintColor = nil
                    expandableBannerViewData = nil
                    bannerViewData = BannerViewData(
                        title: .text(L10n.sponsored),
                        overridedBackgroundColor: .inverseWeak
                    )

					default:
						relationshipInfosViewData = nil
						relationShipInfosTextColor = nil
						relationShipInfosTintColor = nil
						expandableBannerViewData = expandableBanner
						bannerViewData = nil
				}
			} else {
				expandableBannerViewData = expandableBanner
				bannerViewData = nil
				relationshipInfosViewData = nil
				relationShipInfosTextColor = nil
				relationShipInfosTintColor = nil
			}

			if !isFromConversation, let nbCrossing = otherUser.nbCrossings, nbCrossing >= 1 {
				crossingInfosViewData = LabelWithIconViewData(
					text: Language.string(
						fromDate: otherUser.lastMeetDate ?? Date(),
						fullWording: true,
						isMale: otherUser.isLocalizedMale
					),
					numberOfLines: 1,
					leftIcon: IconAsset.walk
				)
			} else {
				crossingInfosViewData = nil
			}

			if !isFromConversation,
			   let activityDate = otherUser.modificationDate {
				let state = abs(activityDate.toLocalTime().timeIntervalSinceNow).getActivityState()
				if state == .none {
					activityInfosViewData = nil
				} else {
					var stateString = ""
					switch state {
						case .now:
							stateString = L10n.profileViewLastActivityDateNow.applying(
								xibLocInfo: Str2StrXibLocInfo(
									genderOtherIsMale: otherUser.isLocalizedMale
								)
							)

						case .today:
							stateString = L10n.profileViewLastActivityDateToday.applying(
								xibLocInfo: Str2StrXibLocInfo(
									genderOtherIsMale: otherUser.isLocalizedMale
								)
							)

						case .recently:
							stateString = L10n.profileViewLastActivityDateRecently.applying(
								xibLocInfo: Str2StrXibLocInfo(
									genderOtherIsMale: otherUser.isLocalizedMale
								)
							)

						case .none:
							break
					}

					activityInfosViewData = LabelWithIconViewData(
						text: stateString,
						numberOfLines: 1,
						leftIcon: IconAsset.circleSmall
					)
				}
			} else {
				activityInfosViewData = nil
			}
		} else {
			bannerViewData = banner
			expandableBannerViewData = nil
			crossingInfosViewData = nil
			activityInfosViewData = nil
			relationshipInfosViewData = nil
			relationShipInfosTextColor = nil
			relationShipInfosTintColor = nil
		}
	}
}
