//
//  ProfileActionButtonsUseCase.swift
//  happnTimeline
//
//  Created by Luc-Olivier Merson on 16/04/2025.
//

import Foundation

import happnConfiguration
import happnPolis
import happnShops
import happnUsers

public enum ProfileActionButtonsType {
	case vibes
	case timeline
	case openProfile
	case supercrush
	case map
}

public protocol ProfileActionButtonsUseCaseProtocol {
	func current(for type: ProfileActionButtonsType, otherUser: UserDomainModel) -> [UserProfileViewInputType]
}

public class ProfileActionButtonsUseCase: ProfileActionButtonsUseCaseProtocol {

	// MARK: - Private properties
	private let boostWorker: BoostWorkerProtocol
	private let userMeUseCase: GetUserMeUseCaseProtocol
	private let actionButtonUseCase: ActionButtonUseCaseProtocol
	private let isPrioritaryLikeActivatedUseCase: IsPrioritaryLikeActivatedUseCaseProtocol

	// MARK: - Initialization
	public init(
		boostWorker: BoostWorkerProtocol,
		userMeUseCase: GetUserMeUseCaseProtocol,
		actionButtonUseCase: ActionButtonUseCaseProtocol,
		isPrioritaryLikeActivatedUseCase: IsPrioritaryLikeActivatedUseCaseProtocol
	) {
		self.boostWorker = boostWorker
		self.userMeUseCase = userMeUseCase
		self.actionButtonUseCase = actionButtonUseCase
		self.isPrioritaryLikeActivatedUseCase = isPrioritaryLikeActivatedUseCase
	}

	// MARK: - Public methods
	public func current(for type: ProfileActionButtonsType, otherUser: UserDomainModel) -> [UserProfileViewInputType] {
		let userMe = userMeUseCase.current()
		let isPrioritaryLikeActivated = isPrioritaryLikeActivatedUseCase.isActivated
		let isMe = userMeUseCase.current().id == otherUser.id

		guard !isMe else { return [] }
		switch type {
			case .vibes:
				return createVibesActionButtons(
					isPrioritaryLikeActivated: isPrioritaryLikeActivated,
					userMe: userMe
				)

			case .timeline:
				return createTimelineActionButtons(
					isPrioritaryLikeActivated: isPrioritaryLikeActivated,
					userMe: userMe,
					isSponsored: otherUser.type == .sponsor
				)

			case .openProfile:
				return createOpenProfileActionButtons(
					otherUser: otherUser,
					isPrioritaryLikeActivated: isPrioritaryLikeActivated,
					userMe: userMe
				)

			case .supercrush:
				return createSuperCrushActionButtons(
					otherUser: otherUser,
					isPrioritaryLikeActivated: isPrioritaryLikeActivated,
					userMe: userMe
				)

			case .map:
				return createMapActionButtons(
					otherUser: otherUser,
					isPrioritaryLikeActivated: isPrioritaryLikeActivated,
					userMe: userMe
				)
		}
	}

	// MARK: - Private methods
	private func createVibesActionButtons(
		isPrioritaryLikeActivated: Bool,
		userMe: UserDomainModel
	) -> [UserProfileViewInputType] {
		return [
			.button(type: .reject(shouldBeLarge: false)),
			.button(type: .like(isPrioritaryLikeActivated: isPrioritaryLikeActivated)),
			.button(type: .superCrush(isSuperCrushCreditRemaining: userMe.flashnoteBalance?.total ?? 0 > 0)),
		]
	}

	private func createTimelineActionButtons(
		isPrioritaryLikeActivated: Bool,
		userMe: UserDomainModel,
        isSponsored: Bool
	) -> [UserProfileViewInputType] {
		let isBoostCreditRemaining = (userMe.boostBalance?.total ?? 0) > 0
		var boostState: BoostState

		if let boost = boostWorker.getCurrentBoost() {
			boostState = boost.status == .running ? .running : (isBoostCreditRemaining ? .available : .default)
		} else {
			boostState = isBoostCreditRemaining ? .available : .default
		}

        var buttons: [UserProfileViewInputType] = [
            .button(type: .reject(shouldBeLarge: true)),
            .button(type: .like(isPrioritaryLikeActivated: isPrioritaryLikeActivated)),
        ]

        if !isSponsored {
            buttons.insert(.button(type: .boost(state: boostState)), at: 0)
            buttons.append(.button(type: .superCrush(isSuperCrushCreditRemaining: userMe.flashnoteBalance?.total ?? 0 > 0)))
        }

        return buttons
	}

	private func createOpenProfileActionButtons(
		otherUser: UserDomainModel,
		isPrioritaryLikeActivated: Bool,
		userMe: UserDomainModel
	) -> [UserProfileViewInputType] {
		let actionButtonConfigurationType = actionButtonUseCase.getActionButtonType(for: otherUser, userMe: userMe)
		let cta = CTAProfileDisplayViewData(
			actionButtonConfigurationType: actionButtonConfigurationType,
			flashnoteBalance: userMe.flashnoteBalance?.total ?? 0,
			isPrioritaryLikeActivated: isPrioritaryLikeActivated
		)
		return [
			cta.isRejectButtonEnabled ? .button(type: .reject(shouldBeLarge: false)) : nil,
			cta.isLikeButtonEnabled ? .button(type: .like(isPrioritaryLikeActivated: isPrioritaryLikeActivated)) : nil,
			cta.isflashnoteButtonEnabled ? .button(type: .superCrush(isSuperCrushCreditRemaining: userMe.flashnoteBalance?.total ?? 0 > 0)) : nil
		].compactMap { $0 }
	}

	private func createSuperCrushActionButtons(
		otherUser: UserDomainModel,
		isPrioritaryLikeActivated: Bool,
		userMe: UserDomainModel
	) -> [UserProfileViewInputType] {
		let actionButtonConfigurationType = actionButtonUseCase.getActionButtonType(for: otherUser, userMe: userMe)
		let cta = CTAProfileDisplayViewData(
			actionButtonConfigurationType: actionButtonConfigurationType,
			flashnoteBalance: userMe.flashnoteBalance?.total ?? 0,
			isPrioritaryLikeActivated: isPrioritaryLikeActivated
		)
		return [
			cta.isRejectButtonEnabled ? .button(type: .reject(shouldBeLarge: false)) : nil,
			cta.isLikeButtonEnabled ? .button(type: .like(isPrioritaryLikeActivated: isPrioritaryLikeActivated)) : nil
		].compactMap { $0 }
	}

	private func createMapActionButtons(
		otherUser: UserDomainModel,
		isPrioritaryLikeActivated: Bool,
		userMe: UserDomainModel
	) -> [UserProfileViewInputType] {
		let actionButtonConfigurationType = actionButtonUseCase.getActionButtonType(for: otherUser, userMe: userMe)
		let cta = CTAProfileDisplayViewData(
			actionButtonConfigurationType: actionButtonConfigurationType,
			flashnoteBalance: userMe.flashnoteBalance?.total ?? 0,
			isPrioritaryLikeActivated: isPrioritaryLikeActivated
		)
		return [
			cta.isRejectButtonEnabled ? .button(type: .reject(shouldBeLarge: false)) : nil,
			cta.isLikeButtonEnabled ? .button(type: .like(isPrioritaryLikeActivated: isPrioritaryLikeActivated)) : nil,
			cta.isflashnoteButtonEnabled ? .button(type: .superCrush(isSuperCrushCreditRemaining: userMe.flashnoteBalance?.total ?? 0 > 0)) : nil
		].compactMap { $0 }
	}
}
