//
//  UserProfilesViewUseCase+PictureContentViewDatas.swift
//  happnProfileUtils
//
//  Created by Safia Chmiti on 18/12/2025.
//
import Foundation

import happnConstants
import happnFlashnote
import happnPolis
import happnTraits
import happnUsers
import happnUtils
import XibLoc

typealias PictureContentBottomViewDataTracking = (viewData: PictureContentBottomViewData, tracking: PictureContentTracking)

public enum PictureContentTracking: String {
	case supercrush
	case roma
	case teaser1 = "teaser_1"
	case teaser2 = "teaser_2"
	case teaser3 = "teaser_3"
	case hobbies
	case spots
	case description
}

public struct PictureContentData: Equatable {
	public let viewData: PictureContentViewDataState
	public let trackingData: PictureContentTracking?

	init(viewData: PictureContentViewDataState, trackingData: PictureContentTracking?) {
		self.viewData = viewData
		self.trackingData = trackingData
	}
}

extension UserProfilesViewUseCase {

	func createContentOnCarousselCardViewData(
		profile: UserDomainModel,
		userMe: UserDomainModel,
		getTraitsUseCase: GetTraitsUseCaseProtocol,
		traits: [TraitDomainModel],
		flashnoteDomainModel: FlashnoteDomainModel?,
		isFromConversation: Bool
	) -> [PictureContentData] {

		let sponsoredBottomElements = buildSponsoredProfileBottomElements(profile: profile)
		let clientProfilBottomElements = buildBottomElements(
			profile: profile,
			userMe: userMe,
			getTraitsUseCase: getTraitsUseCase,
			traits: traits,
			flashnote: flashnoteDomainModel
		)

		let bottomElements = isProfileSponsored ? sponsoredBottomElements : clientProfilBottomElements

		return profile.pictures?
			.indices
			.compactMap { index in
				let bottom = bottomElements[safe: index]

				let viewData: PictureContentViewData?

				switch index {
					case 0:
						viewData = firstPicture(
							user: profile,
							userMe: userMe,
							isFromConversation: isFromConversation,
							bottomData: bottom?.viewData
						)

					case 1:
						viewData = secondPicture(
							user: profile,
							userMe: userMe,
							getTraitsUseCase: getTraitsUseCase,
							bottomData: bottom?.viewData
						)

					default:
						viewData = simplePicture(
							user: profile,
							index: index,
							bottomData: bottom?.viewData
						)
				}

				guard let viewData else {
					return PictureContentData(
						viewData: .empty,
						trackingData: nil
					)
				}
				return PictureContentData(
					viewData: .loaded(viewData: viewData),
					trackingData: bottom?.tracking
				)
			} ?? []
	}

	// MARK: - Content Builders
	private func firstPicture(
		user: UserDomainModel,
		userMe: UserDomainModel,
		isFromConversation: Bool,
		bottomData: PictureContentBottomViewData?
	) -> PictureContentViewData? {

		guard let url = user.picture?.url() else { return nil }

		let topSponsoredPillViewData = PictureContentTopViewData(
			topPillViewData: PillViewData(text: L10n.sponsored),
			topPillStyle: .light
		)

		let topViewData = PictureContentTopViewData(
			topPillViewData: relationToMePill(profile: user, userMe: userMe),
			topPillStyle: .dark
		)

		let centerViewData = PictureContentCenterViewData(
			pillViewData: isProfileSponsored ? nil : crossingOrActivityPill(profile: user, isFromConversation: isFromConversation),
			titleLabelWithIconViewData: mainUserInfo(profile: user),
			infosLabelWithIconViewData: isProfileSponsored ? nil : infoLabels(user: user)
		)

		return PictureContentViewData(
			source: .remote(url),
			topViewData: isProfileSponsored ? topSponsoredPillViewData : topViewData,
			centerViewData: centerViewData,
			bottomViewData: bottomData
		)
	}

	private func secondPicture(
		user: UserDomainModel,
		userMe: UserDomainModel,
		getTraitsUseCase: GetTraitsUseCaseProtocol,
		bottomData: PictureContentBottomViewData?
	) -> PictureContentViewData? {

		guard let url = user.pictures?[safe: 1]?.url() else { return nil }

		let relationshipTraits = relationshipInfoLabels(
			user: user,
			userMe: userMe,
			getTraitsUseCase: getTraitsUseCase
		)

		let heightInfo = heightInfoLabel(
			user: user,
			getTraitsUseCase: getTraitsUseCase
		)

		var infosLabelWithIconViewData: [LabelWithIconViewData]?
		if let relationshipInfo = relationshipTraits {
			infosLabelWithIconViewData = [relationshipInfo]
		}

		if let height = heightInfo {
			infosLabelWithIconViewData?.append(height)
		}

		let centerViewData = PictureContentCenterViewData(
			pillViewData: nil,
			titleLabelWithIconViewData: mainUserInfo(profile: user),
			infosLabelWithIconViewData: isProfileSponsored ? nil : infosLabelWithIconViewData
		)

		return PictureContentViewData(
			source: .remote(url),
			topViewData: nil,
			centerViewData: centerViewData,
			bottomViewData: isProfileSponsored ? nil : bottomData
		)
	}

	private func simplePicture(
		user: UserDomainModel,
		index: Int,
		bottomData: PictureContentBottomViewData?
	) -> PictureContentViewData? {

		guard let url = user.pictures?[safe: index]?.url() else { return nil }

		let centerViewData = PictureContentCenterViewData(
			pillViewData: nil,
			titleLabelWithIconViewData: mainUserInfo(profile: user),
			infosLabelWithIconViewData: []
		)

		return PictureContentViewData(
			source: .remote(url),
			topViewData: nil,
			centerViewData: centerViewData,
			bottomViewData: isProfileSponsored ? nil : bottomData
		)
	}

	// MARK: - Top Elements

	// MARK: - Relation To Me Info
	private func relationToMePill(profile: UserDomainModel, userMe: UserDomainModel) -> PillViewData? {
		var text: String?
		var iconToken: IconAssetToken?
		var overridedIconTintColor: PolisColor.IconToken?

		let userMeIsMale = userMe.isLocalizedMale
		let otherUserIsMale = profile.isLocalizedMale

		if profile.hasCharmedMe {
			text = L10n.profileViewSupercrushReceived.applying(
				xibLocInfo: Str2StrXibLocInfo(
					genderMeIsMale: userMeIsMale,
					genderOtherIsMale: otherUserIsMale
				))
			iconToken = IconAsset.supercrush
			overridedIconTintColor = .skyDefault
		} else if profile.hasAcceptedMe && userMe.isSubscriber && profile.relationToMe.isNil {
			text = L10n.reactionReceivedText2.applying(
				xibLocInfo: Str2StrXibLocInfo(
					genderMeIsMale: userMeIsMale,
					genderOtherIsMale: otherUserIsMale
				)
			)
			iconToken = IconAsset.heart
			overridedIconTintColor = .actionLike
		} else if profile.isCharmed ?? false {
			text = L10n.profileViewSupercrushSent.applying(
				xibLocInfo: Str2StrXibLocInfo(
						  genderMeIsMale: userMeIsMale,
						  genderOtherIsMale: otherUserIsMale
					  )
				  )
			iconToken = IconAsset.supercrush
			overridedIconTintColor = .skyDefault
		} else {
			switch profile.relationToMe {
				case .accepted:
					text = L10n.profileViewLikeSent.applying(
						xibLocInfo: Str2StrXibLocInfo(
							genderMeIsMale: userMeIsMale,
							genderOtherIsMale: otherUserIsMale
						)
					)
					iconToken = IconAsset.heart
					overridedIconTintColor = .actionLike

				default:
					break
			}
		}

		guard let iconToken = iconToken else { return nil }
		return PillViewData(
			text: text,
			leftIcon: PillIconType.localIcon(iconToken),
			overridedIconTintColor: overridedIconTintColor
		)
	}

	// MARK: - Crossing Or Activity Info
	private func crossingOrActivityPill(profile: UserDomainModel, isFromConversation: Bool) -> PillViewData? {
		guard !isFromConversation, !isMe else { return nil }

		if let crossings = profile.nbCrossings, crossings >= 1, !locationConfigurationUseCase.current.mapAndElementsDisabled {
			let text = Language.string(
				fromDate: profile.lastMeetDate ?? Date(),
				fullWording: true,
				isMale: profile.isLocalizedMale
			)
			return PillViewData(text: text, leftIcon: .localIcon(IconAsset.walk))
		}

		guard let date = profile.modificationDate else { return nil }

		let state = abs(date.toLocalTime().timeIntervalSinceNow).getActivityState()
		guard state != .none else { return nil }

		let localizedText: String = {
			switch state {
			case .now:
				return L10n.profileViewLastActivityDateNow

			case .today:
				return L10n.profileViewLastActivityDateToday

			case .recently:
				return L10n.profileViewLastActivityDateRecently

			default:
				return ""
			}
		}().applying(
			xibLocInfo: Str2StrXibLocInfo(genderOtherIsMale: profile.isLocalizedMale)
		)

		return PillViewData(
			text: localizedText,
			leftIcon: .localIcon(IconAsset.circle),
			overridedIconTintColor: .mantis
		)
	}

	// MARK: - Middle Infos

	// MARK: - FirstName and Certification
	private func mainUserInfo(profile: UserDomainModel) -> LabelWithIconViewData? {
		guard let firstName = profile.firstName else { return nil }
		return LabelWithIconViewData(
			nameAndAge: firstName,
			isCertified: profile.isCertified
		)
	}

	// MARK: - Age and City of Residence
	private func infoLabels(user: UserDomainModel) -> [LabelWithIconViewData] {
		var labels: [LabelWithIconViewData] = []

		if let age = user.age {
			labels.append(
				LabelWithIconViewData(
					text: L10n.profileViewAgeInformation.applying(xibLocInfo: Str2StrXibLocInfo(pluralValue: NumberAndFormat(age))),
					leftIcon: IconAsset.cake
				)
			)
		}

		if let city = user.cityOfResidence?.city {
			let text = L10n.profileViewCityInformation.applying(xibLocInfo: Str2StrXibLocInfo(replacement: city))
			labels.append(
				LabelWithIconViewData(
					text: text,
					leftIcon: IconAsset.home
				)
			)
		}

		return labels
	}

	// MARK: - Relationships
	private func relationshipInfoLabels(
		user: UserDomainModel,
		userMe: UserDomainModel,
		getTraitsUseCase: GetTraitsUseCaseProtocol
	) -> LabelWithIconViewData? {

		let relationshipAnswers = UserTraitMultiChoiceViewData
			.multiChoiceTraitAnswers(
				trait: getTraitsUseCase.current(trait: .relationship),
				targetUser: user,
				userMe: user,
				common: false
			)

		let myRelationshipAnswers =  UserTraitMultiChoiceViewData
			.multiChoiceTraitAnswers(
				trait: getTraitsUseCase.current(trait: .relationship),
				targetUser: userMe,
				userMe: userMe,
				common: false
			)

		let finalArray = relationshipAnswers.sorted { myRelationshipAnswers.contains($0) && !myRelationshipAnswers.contains($1) }

		guard finalArray.isNotEmpty else { return nil }
		return LabelWithIconViewData(text: finalArray.joined(separator: "  \u{00B7}  "), numberOfLines: 2, leftIcon: IconAsset.search)
	}

	// MARK: - Height Info
	private func heightInfoLabel(
		user: UserDomainModel,
		getTraitsUseCase: GetTraitsUseCaseProtocol
	) -> LabelWithIconViewData? {

		let viewDatas = TraitAndAnswerViewData.getTraitsWithAnswersViewData(
			isUserMeMale: false,
			traits: getTraitsUseCase.current() ?? [],
			traitsAnswers: user.traits
		)

		guard
			let heightData = viewDatas.first(where: {
				$0.trait.id == Constants.Features.TraitMultiChoice.heightTraitId
			}),
			let text: String = {
				switch heightData.answer {
					case .floatRange(let value):
						return value

					default:
						return nil
				}
			}()
		else { return nil }

		return LabelWithIconViewData(text: text, leftIcon: IconAsset.ruler)
	}

	// MARK: - Bottom Elements
	private func buildSponsoredProfileBottomElements(profile: UserDomainModel) -> [PictureContentBottomViewDataTracking] {
		var elements: [PictureContentBottomViewData] = []
		var trackingElements: [PictureContentTracking] = []

		if let descriptionElement = buildDescriptionElement(profile: profile) {
			elements.append(descriptionElement)
			trackingElements.append(.description)
		}

		return Array.zip(elements, trackingElements)
	}

	private func buildBottomElements(
		profile: UserDomainModel,
		userMe: UserDomainModel,
		getTraitsUseCase: GetTraitsUseCaseProtocol,
		traits: [TraitDomainModel],
		flashnote: FlashnoteDomainModel?
	) -> [PictureContentBottomViewDataTracking] {

		var elements: [PictureContentBottomViewData] = []
		var trackingElements: [PictureContentTracking] = []

		// MARK: - Flashnote OR Roma badge
		if let flashnotePreview = flashnote?.preview {
			let avatarViewData = AvatarIconViewData(type: .icon(IconAsset.supercrush), overridedTint: .skyDefault)

			elements.append(
				PictureContentBottomViewData(
					bannerType: .banner(
						BannerViewData(
							avatarIconViewData: avatarViewData,
							message: flashnotePreview,
							extendFrame: true
						),
						.opaque
					)
				)
			)
			trackingElements.append(.supercrush)
		}

		if flashnote == nil,
		   let romaBadge = createExpandableBannerViewData(
			getTraitsUseCase: getTraitsUseCase,
			traits: traits,
			userMe: userMe,
			profile: profile
		   ) {
			elements.append(PictureContentBottomViewData(bannerType: .expandableBanner(romaBadge, .opaque)))
			trackingElements.append(.roma)

		}

		// MARK: - First Teaser
		if let firstTeaser = returnTeaserElement(profile: profile, traits: traits, index: 0) {
			elements.append(firstTeaser)
			trackingElements.append(.teaser1)
		}

		// MARK: - Hobbies
		let hobbies = UserTraitMultiChoiceViewData.multiChoiceTraitAnswers(
			trait: getTraitsUseCase.current(trait: .hobbies),
			targetUser: profile,
			userMe: userMe,
			common: false
		)

		let userHobbies = UserTraitMultiChoiceViewData.multiChoiceTraitAnswers(
			trait: getTraitsUseCase.current(trait: .hobbies),
			targetUser: userMe,
			userMe: userMe,
			common: false
		)

		let sortedHobbies = hobbies.sorted { userHobbies.contains($0) && !userHobbies.contains($1) }

		if hobbies.isNotEmpty {
			elements.append(
				PictureContentBottomViewData(
					bannerType: .expandableBanner(
						ExpandableBannerViewData(
							title: L10n.hobbiesProfileDisplayEntryTitle,
							pillViewData: sortedHobbies.map { PillViewData(text: $0, isCommonAnswer: isMe ? false : userHobbies.contains($0)) }
						)
					)
				)
			)
			trackingElements.append(.hobbies)
		}

		// MARK: - Spots
		if let spots = profile.spots, spots.isNotEmpty {
			let userSpots = userMe.spots ?? []
			let sortedSpots = spots.sorted { userSpots.contains($0) && !userSpots.contains($1) }
			elements.append(
				PictureContentBottomViewData(
					bannerType: .expandableBanner(
						ExpandableBannerViewData(
							title: L10n.dailyProfileViewSpotsTitle,
							pillViewData: sortedSpots.map {
								PillViewData(text: $0.name, isCommonAnswer: isMe ? false : userSpots.contains($0))
							}
						)
					)
				)
			)
			trackingElements.append(.spots)
		}

		// MARK: - Second Teaser
		if let secondTeaser = returnTeaserElement(profile: profile, traits: traits, index: 1) {
			elements.append(secondTeaser)
			trackingElements.append(.teaser2)
		}

		// MARK: - Third Teaser
		if let thirdTeaser = returnTeaserElement(profile: profile, traits: traits, index: 2) {
			elements.append(thirdTeaser)
			trackingElements.append(.teaser3)
		}

		// MARK: - Description
		if let descriptionElement = buildDescriptionElement(profile: profile) {
			elements.append(descriptionElement)
			trackingElements.append(.description)
		}

		return Array.zip(elements, trackingElements)
	}

	private func buildDescriptionElement(profile: UserDomainModel) -> PictureContentBottomViewData? {
		guard let description = profile.description, !description.isEmpty else { return nil }
		return PictureContentBottomViewData(
			bannerType: .banner(
				BannerViewData(
					message: description,
					extendFrame: true,
					overridedMessageNumberOfLines: 7,
				)
			)
		)
	}

	private func returnTeaserElement(
		profile: UserDomainModel,
		traits: [TraitDomainModel]?,
		index: Int
	) -> PictureContentBottomViewData? {
		guard let teaser = getTeaserQuestionAndAnswer(
			profile: profile,
			traits: traits,
			index: index
		) else { return nil }

		return PictureContentBottomViewData(
			bannerType: .banner(
				BannerViewData(
					leftIcon: .token(IconAsset.quoteOpen),
					title: .text(teaser.0),
					message: teaser.1,
					extendFrame: true,
					overridedTitleFont: PolisFont.Caption2.regular,
					overridedMessageFont: PolisFont.Body.medium,
					overridedMessageNumberOfLines: 7
				)
			)
		)
	}

	private func getTeaserQuestionAndAnswer(
		profile: UserDomainModel,
		traits: [TraitDomainModel]?,
		index: Int
	) -> (String, String)? {
		guard let userInputs = (profile.traits?.filter { $0.type == .userInput }) else { return nil }

		guard let userInput = userInputs.first,
			  case let .userInput(answers) = userInput.answer,
			  let answer = answers[safe: index],
			  let trait = (traits?.first { $0.id == userInput.traitId }),
			  case let .userInput(categories) = trait.options
		else { return nil }

		let options = categories.flatMap { $0.options }
		guard let option = (options.first { $0.id == answer.optionId }) else { return nil }

		return (option.text, answer.answerText)
	}
}
