//
//  CardCrossingViewData+Timeline.swift
//
//
//  Created by Nicolas Sabella on 12/05/2023.
//

import Foundation
import MapKit

import XibLoc

import happnConstants
import happnPolis
import happnUtils

public enum CardCrossingType: Equatable {
	case location(placemark: CLPlacemark)
	case locationDisable
	case locationUnavailable

	public var placemark: CLPlacemark? {
		switch self {
			case let .location(placemark):
				return placemark

			case .locationDisable, .locationUnavailable:
				return nil
		}
	}
}

extension CardCrossingViewData {
	public init(
		type: CardCrossingType,
		userName: String,
		nbCrossing: Int,
		crossingDate: Date?,
		placemark: CLPlacemark?,
		userIsMale: Bool,
		otherUserIsMale: Bool,
		userMePicture: URL?,
		otherUserPicture: URL?
	) {
		let title = {
			if nbCrossing <= Constants.Features.OpenProfile.nbCrossingThreshold {
				L10n.profileCrossingsMapNumberOfCrossings
					.applying(
						xibLocInfo: Str2StrXibLocInfo(
							replacement: userName,
							pluralValue: NumberAndFormat(max(1, nbCrossing)),
							genderMeIsMale: userIsMale,
							genderOtherIsMale: otherUserIsMale
						)
					)
			} else {
				L10n.profileCrossingsMapSeveralCrossings.applying(xibLocInfo: Str2StrXibLocInfo(replacement: userName))
			}
		}()

		var buttonText = ""
		var dateText = Language.string(fromDate: crossingDate ?? Date())
		var descriptionTitle = ""
		var descriptionButtonText = ""

		switch type {
			case .location:
				if let crossingAddress = placemark?.crossingAddressString(),
				   !crossingAddress.isEmpty {
					dateText += " " + L10n.profileCrossingsNear
					buttonText = crossingAddress
				}

			case .locationDisable:
				buttonText = L10n.profileViewCrossingsSomewhereOnEarth
				descriptionTitle = L10n.mapPlaceholderUnknowLocationLockedByMeText
					.applying(
						xibLocInfo: Str2StrXibLocInfo(
							genderMeIsMale: userIsMale,
							genderOtherIsMale: otherUserIsMale
						)
					)
				descriptionButtonText = L10n.mapPlaceholderUnknowLocationLockedByMeButton

			case .locationUnavailable:
				buttonText = L10n.profileViewCrossingsSomewhereOnEarth
				descriptionTitle = L10n.mapPlaceholderUnknowLocationLockedByOther
		}

		self.init(
			leftAvatarIconURL: userMePicture,
			rightAvatarIconURL: otherUserPicture,
			titleText: title,
			dateText: dateText,
			buttonText: buttonText,
			descriptionText: descriptionTitle,
			descriptionButtonText: descriptionButtonText.isEmpty ? nil : descriptionButtonText
		)
	}
}
