//
//  ProfileDisplayRewindUseCase.swift
//  happnProfileUtils
//
//  Created by Luc-Olivier Merson on 19/11/2025.
//

import RxCocoa
import RxSwift

import happnTracking
import happnUsers

public protocol ProfileDisplayRewindUseCaseProtocol {
	var current: UserDomainModel? { get }
	func rewind(screen: ScreenName) -> Completable
	func returnRewindedUser() -> Observable<UserDomainModel>
}

public class ProfileDisplayRewindUseCase: ProfileDisplayRewindUseCaseProtocol {

	// MARK: - Private properties
	private let unrejectUseCase: WrapperUnRejectUserUseCaseProtocol
	private let rewindRepository: ProfileDisplayRewindRepositoryProtocol
	private let repositories: [DidRewindUserProtocol]
	private let tracker: AppTrackerProtocol

	// MARK: - Public properties
	public var current: UserDomainModel? {
		return rewindRepository.get()
	}

	// MARK: - Initialization
	public init(
		unrejectUseCase: WrapperUnRejectUserUseCaseProtocol,
		rewindRepository: ProfileDisplayRewindRepositoryProtocol,
		repositories: [DidRewindUserProtocol],
		tracker: AppTrackerProtocol
	) {
		self.unrejectUseCase = unrejectUseCase
		self.rewindRepository = rewindRepository
		self.repositories = repositories
		self.tracker = tracker
	}

	// MARK: - Public methods
	public func rewind(screen: ScreenName) -> Completable {
		rewindRepository.observe()
			.compactMap { $0 }
			.take(1)
			.do(onNext: { [weak self] profile in
				self?.repositories.forEach { $0.didRewind(user: profile) }
				self?.rewindRepository.remove()
				self?.tracker.sendEvent(event: .rewindUser(screen: screen, receiverId: profile.id))
			})
			.flatMap { [weak self] profile -> Observable<Void> in
				self?.unrejectUseCase.unreject(user: profile, origin: .timeline).asVoidObservable() ?? .just(())
			}
			.ignoreElements()
			.asCompletable()
	}

	public func returnRewindedUser() -> Observable<UserDomainModel> {
		return rewindRepository.observe().compactMap { $0 }.take(1)
	}
}
